﻿<#
.SYNOPSIS
  WordPress BackWPupのバックアップファイルをサーバーからローカルにダウンロードします。

.DESCRIPTION
  listコマンドでサーバー上のファイル一覧を取得し、getコマンドでローカルに存在しないファイルをダウンロードします。

.PARAMETER TargetUrl
  backwpup-download.php のフルURL (例: https://example.com/backwpup-download.php)

.PARAMETER AccessKey
  サーバー側PHPスクリプトで設定した秘密のアクセスキー

.PARAMETER DownloadPath
  バックアップファイルを保存するローカルディレクトリのパス
#>
param(
    [Parameter(Mandatory=$true)]
    [string]$TargetUrl,

    [Parameter(Mandatory=$true)]
    [string]$AccessKey,

    [Parameter(Mandatory=$true)]
    [string]$DownloadPath
)

# ダウンロードディレクトリの確認と作成
if (-not (Test-Path -Path $DownloadPath -PathType Container)) {
    Write-Host "ダウンロードディレクトリ $DownloadPath が存在しないため作成します。"
    try {
        New-Item -Path $DownloadPath -ItemType Directory | Out-Null
    } catch {
        Write-Error "ダウンロードディレクトリの作成に失敗しました: $($_.Exception.Message)"
        exit 1
    }
}

# --- 1. サーバー上のファイル一覧を取得 (listコマンド) ---
Write-Host "サーバー上のファイル一覧を取得中..."
try {
    $ListBody = @{
        'access_key' = $AccessKey;
        'command'    = 'list'
    }

    $ListResponse = Invoke-RestMethod -Uri $TargetUrl -Method Post -Body $ListBody -ContentType 'application/x-www-form-urlencoded'

    if ($ListResponse.status -ne 'success') {
        Write-Error "サーバーからのリスト取得に失敗しました: $($ListResponse | ConvertTo-Json -Depth 1)"
        exit 1
    }

    $ServerFiles = $ListResponse.files
    Write-Host "サーバー上で合計 $($ServerFiles.Count) 個のファイルが見つかりました。"
} catch {
    Write-Error "listコマンドの実行中にエラーが発生しました: $($_.Exception.Message)"
    exit 1
}

# --- 2. ローカルディレクトリのファイル一覧を取得 ---
Write-Host "ローカルディレクトリのファイル一覧を取得中..."
$LocalFiles = Get-ChildItem -Path $DownloadPath -Filter "*.zip" | Select-Object -ExpandProperty Name
Write-Host "ローカルで合計 $($LocalFiles.Count) 個のファイルが見つかりました。"


# --- 3. 差分をダウンロード (getコマンド) ---
$FilesToDownload = $ServerFiles | Where-Object { $_ -notin $LocalFiles }

if ($FilesToDownload.Count -eq 0) {
    Write-Host "ダウンロードすべき新しいファイルはありませんでした。"
    exit 0
}

Write-Host "新しくダウンロードすべきファイルは $($FilesToDownload.Count) 個です。"

$FilesToDownload | ForEach-Object {
    $FileName = $_
    $LocalFilePath = Join-Path -Path $DownloadPath -ChildPath $FileName

    Write-Host "-> $FileName をダウンロード中..."

    try {
        $GetBody = @{
            'access_key' = $AccessKey;
            'command'    = 'get';
            'filename'   = $FileName
        }

        # ダウンロードを実行
        Invoke-WebRequest -Uri $TargetUrl -Method Post -Body $GetBody -OutFile $LocalFilePath -ContentType 'application/x-www-form-urlencoded'

        # ファイルサイズチェックなど、必要に応じて追加の検証をここで行うことができます。
        
        Write-Host "   $FileName のダウンロードが完了しました。"
    } catch {
        Write-Error "ファイルのダウンロード中にエラーが発生しました ($FileName): $($_.Exception.Message)"
        # エラーが発生しても処理を続行
    }
}

Write-Host "すべての処理が完了しました。"
