﻿using System.Diagnostics;

namespace Asynchronization;

internal static class Program
{
    private const string BaseUrl = "https://www.north-innovations.com/wp-content/scripts/async-demo.php";
    private static readonly HttpClient _httpClient = new();
    private static readonly Stopwatch _stopwatch = new();

    public static async Task Main()
    {
        // 同期
        await MeasureAsync("同期ダウンロード", async () =>
        {
            Download("A");
            Download("B");
            Download("C");
        });

        // 非同期
        await MeasureAsync("非同期ダウンロード", async () =>
        {
            var taskA = DownloadAsync("A");
            var taskB = DownloadAsync("B");
            var taskC = DownloadAsync("C");
            await Task.WhenAll(taskA, taskB, taskC);
        });

        // 同期コンテキストあり、非同期
        GuiLikeEventLoop.Run(() => MeasureAsync("同期コンテキストあり（GUI環境を再現）で非同期ダウンロード", async () =>
        {
            var taskA = DownloadAsync("A");
            var taskB = DownloadAsync("B");
            var taskC = DownloadAsync("C");
            await Task.WhenAll(taskA, taskB, taskC);
        }));

        // 別スレッド、同期
        await MeasureAsync("別スレッドで同期ダウンロード", async () =>
        {
            var taskA = Task.Run(() => Download("A"));
            var taskB = Task.Run(() => Download("B"));
            var taskC = Task.Run(() => Download("C"));
            await Task.WhenAll(taskA, taskB, taskC);
        });

        // 別スレッド、非同期
        await MeasureAsync("別スレッドで非同期ダウンロード", async () =>
        {
            var taskA = Task.Run(() => DownloadAsync("A"));
            var taskB = Task.Run(() => DownloadAsync("B"));
            var taskC = Task.Run(() => DownloadAsync("C"));
            await Task.WhenAll(taskA, taskB, taskC);
        });
    }

    // 同期ダウンロード
    private static void Download(string id)
    {
        try
        {
            var url = $"{BaseUrl}?id={id}";
            WriteProgress($"接続 Id={id} Url={url}");
            using var stream = _httpClient.GetStreamAsync(url).Result;
            using var reader = new StreamReader(stream);
            while (true)
            {
                var line = reader.ReadLine();
                if (line == null)
                    break;
                WriteProgress($"受信 {line.Replace("<br>", "")}");
            }
            WriteProgress($"切断 Id={id}");
        }
        catch (Exception ex)
        {
            Console.WriteLine(ex.Message);
        }
    }

    // 非同期ダウンロード
    private static async Task DownloadAsync(string id)
    {
        try
        {
            var url = $"{BaseUrl}?id={id}";
            WriteProgress($"接続 Id={id} Url={url}");
            using var stream = await _httpClient.GetStreamAsync(url);
            using var reader = new StreamReader(stream);
            while (true)
            {
                var line = await reader.ReadLineAsync();
                if (line == null)
                    break;
                WriteProgress($"受信 {line.Replace("<br>", "")}");
            }
            WriteProgress($"切断 Id={id}");
        }
        catch (Exception ex)
        {
            Console.WriteLine(ex.Message);
        }
    }

    // 計測
    private static async Task MeasureAsync(string description, Func<Task> asyncTask)
    {
        Console.WriteLine($"{description}を開始します...");
        _stopwatch.Restart();
        await asyncTask();
        Console.WriteLine($"全ての処理が完了しました。所要時間: {_stopwatch.Elapsed.TotalSeconds:N2}秒\n");
    }

    // 進捗をコンソールに書き込む
    private static void WriteProgress(string message)
    {
        Console.WriteLine($"Time={_stopwatch.Elapsed.TotalSeconds:N2} ThreadId={Environment.CurrentManagedThreadId} {message}");
    }
}
