﻿using System.Collections.Concurrent;
using System.Diagnostics;

namespace DataRaceAndLock;

internal static class Program
{
    public static void Main()
    {
        // メモリ同時書き込み
        Console.WriteLine("=== メモリ同時書き込み ===");
        foreach (var threadCount in new int[] { 1, 2, 4, 8 })
        {
            var count = 0;
            const int totalCount = 1_000_000;
            int countPerThread = totalCount / threadCount;
            using var barrier = new Barrier(threadCount);
            Parallel.For(0, threadCount, threadId =>
            {
                barrier.SignalAndWait(); // 開始タイミングを合わせる
                for (var index = 0; index < countPerThread; index++)
                {
                    count++;
                }
            });
            Console.WriteLine($"スレッド数={threadCount} 結果={count:N0}");
        }
        Console.WriteLine();

        // メモリ同時書き込みの詳細
        Console.WriteLine("=== メモリ同時書き込みの詳細 ===");
        foreach (var threadCount in new int[] { 1, 4 })
        {
            Console.WriteLine($"--- スレッド数={threadCount} ---");
            Console.WriteLine("メモリから読み込んだ値：");
            const int totalCount = 20;
            const int maxColumnLength = 2;
            var memory = 0;
            int countPerThread = totalCount / threadCount;
            var threadIds = Enumerable.Range(0, threadCount);
            var readValueSet = threadIds.Select(id => new int[countPerThread]).ToArray();
            using var barrier = new Barrier(threadCount);
            Parallel.For(0, threadCount, threadId =>
            {
                var readValues = readValueSet[threadId];
                barrier.SignalAndWait(); // 開始タイミングを合わせる
                for (var index = 0; index < countPerThread; index++)
                {
                    var register = memory; // メモリからレジスタに読み込み
                    readValues[index] = register;
                    Thread.SpinWait(20);
                    register++;            // レジスタの値をインクリメント
                    Thread.SpinWait(20);
                    memory = register;     // メモリにレジスタの値を書き込み
                    Thread.SpinWait(20);
                }
            });
            Console.WriteLine("| " + string.Join(" | ", threadIds.Select(id => $"T{id + 1}".PadRight(maxColumnLength))) + " |");
            Console.WriteLine("+-" + string.Join("-+-", threadIds.Select(id => new string('-', maxColumnLength))) + "-+");
            var query = from threadId in Enumerable.Range(0, threadCount)
                        from readValue in readValueSet[threadId]
                        group threadId by readValue into item
                        orderby item.Key
                        select item;
            foreach (var item in query)
            {
                var columns = Enumerable.Range(0, threadCount).Select(id => new string(' ', maxColumnLength)).ToArray();
                foreach (var threadId in item)
                    columns[threadId] = $"{item.Key,maxColumnLength}";
                Console.WriteLine("| " + string.Join(" | ", columns) + " |");
            }
            Console.WriteLine($"結果={memory:N0}");
            Console.WriteLine();
        }

        // ロックしながらメモリ書き込み
        Console.WriteLine("=== ロックしながらメモリ書き込み ===");
        foreach (var threadCount in new int[] { 1, 2, 4, 8 })
        {
            var stopwatch = Stopwatch.StartNew();
            var count = 0;
            const int totalCount = 1_000_000;
            int countPerThread = totalCount / threadCount;
            var lock1 = new Lock();
            Parallel.For(0, threadCount, threadId =>
            {
                for (var index = 0; index < countPerThread; index++)
                {
                    lock (lock1)
                    {
                        count++;
                    }
                }
            });
            stopwatch.Stop();
            Console.WriteLine($"スレッド数={threadCount} 結果={count:N0} 処理時間={stopwatch.ElapsedMilliseconds}ms");
        }
        Console.WriteLine();

        // インターロックを使ってメモリ書き込み
        Console.WriteLine("=== インターロックを使ってメモリ書き込み ===");
        foreach (var threadCount in new int[] { 1, 2, 4, 8 })
        {
            var stopwatch = Stopwatch.StartNew();
            var count = 0;
            const int totalCount = 1_000_000;
            int countPerThread = totalCount / threadCount;
            Parallel.For(0, threadCount, threadId =>
            {
                for (var index = 0; index < countPerThread; index++)
                {
                    Interlocked.Increment(ref count);
                }
            });
            stopwatch.Stop();
            Console.WriteLine($"スレッド数={threadCount} 結果={count:N0} 処理時間={stopwatch.ElapsedMilliseconds}ms");
        }
        Console.WriteLine();

        // リスト追加
        Console.WriteLine("=== リスト追加 ===");
        foreach (var threadCount in new int[] { 1, 2, 4, 8 })
        {
            try
            {
                var list = new List<byte>();
                var partitioner = Partitioner.Create(0, 1_000_000);
                var options = new ParallelOptions() { MaxDegreeOfParallelism = threadCount };
                Parallel.ForEach(partitioner, options, range =>
                {
                    for (var index = range.Item1; index < range.Item2; index++)
                    {
                        list.Add((byte)(index % 256));
                    }
                });
                Console.WriteLine($"スレッド数={threadCount} 項目の数={list.Count:N0}");
            }
            catch (Exception ex)
            {
                // 稀に例外で落ちることがある。
                Console.WriteLine(ex.Message);
            }
        }
        Console.WriteLine();

        // ロックしながらリスト追加
        Console.WriteLine("=== ロックしながらリスト追加 ===");
        foreach (var threadCount in new int[] { 1, 2, 4, 8 })
        {
            var stopwatch = Stopwatch.StartNew();
            var list = new List<byte>();
            var lock1 = new Lock();
            var partitioner = Partitioner.Create(0, 1_000_000);
            var options = new ParallelOptions() { MaxDegreeOfParallelism = threadCount };
            Parallel.ForEach(partitioner, options, range =>
            {
                for (var index = range.Item1; index < range.Item2; index++)
                {
                    var value = (byte)(index % 256); // ロック時間を最小にするため、先に計算しておく
                    lock (lock1)
                    {
                        list.Add(value);
                    }
                }
            });
            stopwatch.Stop();
            Console.WriteLine($"スレッド数={threadCount} 項目の数={list.Count:N0} 処理時間={stopwatch.ElapsedMilliseconds}ms");
        }
        Console.WriteLine();

        // コンカレントバッグに追加
        Console.WriteLine("=== コンカレントバッグに追加 ===");
        foreach (var threadCount in new int[] { 1, 2, 4, 8 })
        {
            var stopwatch = Stopwatch.StartNew();
            var bag = new ConcurrentBag<byte>();
            var partitioner = Partitioner.Create(0, 1_000_000);
            var options = new ParallelOptions() { MaxDegreeOfParallelism = threadCount };
            Parallel.ForEach(partitioner, options, range =>
            {
                for (var index = range.Item1; index < range.Item2; index++)
                    bag.Add((byte)(index % 256));
            });
            stopwatch.Stop();
            Console.WriteLine($"スレッド数={threadCount} 項目の数={bag.Count:N0} 処理時間={stopwatch.ElapsedMilliseconds}ms");
        }
        Console.WriteLine();

        // 配列にセット
        Console.WriteLine("=== 配列にセット ===");
        foreach (var threadCount in new int[] { 1, 2, 4, 8 })
        {
            var stopwatch = Stopwatch.StartNew();
            const int count = 1_000_000;
            var array = new byte[count];
            var partitioner = Partitioner.Create(0, count);
            var options = new ParallelOptions() { MaxDegreeOfParallelism = threadCount };
            Parallel.ForEach(partitioner, options, range =>
            {
                for (var index = range.Item1; index < range.Item2; index++)
                    array[index] = (byte)(index % 256);
            });
            stopwatch.Stop();
            Console.WriteLine($"スレッド数={threadCount} 処理時間={stopwatch.Elapsed.TotalMilliseconds:N1}ms");
        }
        Console.WriteLine();

        // Task.Run内のラムダ式
        Console.WriteLine("=== Task.Run内のラムダ式 ===");
        {
            // ワーカースレッド数を3に変更
            ThreadPool.GetMinThreads(out var minWorker, out var minPort);
            ThreadPool.GetMaxThreads(out var maxWorker, out var maxPort);
            ThreadPool.SetMinThreads(3, minPort);
            ThreadPool.SetMaxThreads(3, maxPort);

            // 引数の値をコンソールに出力する関数
            var writeConsole = new Action<int>(id =>
            {
                Console.WriteLine($"id={id}");
                Thread.Sleep(50);
            });

            // イテレータの値をそのまま渡した場合
            Console.WriteLine("--- イテレータの値をそのまま渡した場合 ---");
            var tasks = new List<Task>();
            for (var index = 0; index < 10; index++)
            {
                var task = Task.Run(() => writeConsole(index));
                tasks.Add(task);
                Thread.Sleep(1);
            }
            Task.WaitAll(tasks);
            Console.WriteLine();

            // 一度別の変数に代入してから渡した場合
            Console.WriteLine("--- 一度別の変数に代入してから渡した場合 ---");
            tasks.Clear();
            for (var index = 0; index < 10; index++)
            {
                var id = index;
                var task = Task.Run(() => writeConsole(id));
                tasks.Add(task);
                Thread.Sleep(1);
            }
            Task.WaitAll(tasks);
            Console.WriteLine();

            // ワーカースレッド数を戻に戻す
            ThreadPool.SetMinThreads(minWorker, minPort);
            ThreadPool.SetMaxThreads(maxWorker, maxPort);
        }

        // コンパイラの最適化
        Console.WriteLine("=== コンパイラの最適化 ===");
        {
            // コンパイラの最適化状況を表示
#if DEBUG
            Console.WriteLine("現在のビルドはデバッグビルドです。コンパイラの最適化がかかっていません。");
#else
            Console.WriteLine("現在のビルドはリリースビルドです。コンパイラの最適化がかかっています。");
#endif

            // タスクを開始
            var isStopping = false;
            var task = Task.Run(() =>
            {
                var count = 20_000_000_000;
                while (!isStopping)
                {
                    if (--count == 0)
                        return false;
                }
                return true;
            });

            // 少し待ってから停止フラグをセット
            Thread.Sleep(100);
            isStopping = true;

            // 結果を表示
            Console.WriteLine(task.Result ? "正常に停止しました。" : "タイムアウトしました。");
        }
    }
}
